// Simple popup script for Thorium Update Checker
document.addEventListener('DOMContentLoaded', initialize);

// Auto-detect browser version using User-Agent Client Hints API
// Note: Since Chrome 101, the UA string version is FROZEN to X.0.0.0
// We must use the Client Hints API to get the real full version
async function detectBrowserVersion() {
  console.log("Detecting browser version...");
  
  // Method 1: User-Agent Client Hints API (preferred, gives real version)
  if (navigator.userAgentData && navigator.userAgentData.getHighEntropyValues) {
    try {
      const data = await navigator.userAgentData.getHighEntropyValues(['fullVersionList']);
      console.log("Client Hints data:", data);
      
      if (data.fullVersionList && data.fullVersionList.length > 0) {
        // Look for Chromium brand (Thorium is based on Chromium)
        const chromium = data.fullVersionList.find(b => b.brand === 'Chromium');
        if (chromium && chromium.version) {
          console.log("Detected Chromium version via Client Hints:", chromium.version);
          return chromium.version;
        }
        
        // Fallback to first brand with a real version (not 99.0.0.0 placeholder)
        for (const brand of data.fullVersionList) {
          if (brand.version && !brand.version.startsWith('99.') && !brand.brand.includes('Not')) {
            console.log("Detected version via Client Hints:", brand.version);
            return brand.version;
          }
        }
      }
    } catch (e) {
      console.log("Client Hints API error:", e);
    }
  }
  
  // Method 2: Fallback to UA string (will only get major version due to Chrome 101+ freezing)
  const ua = navigator.userAgent;
  console.log("Falling back to User Agent:", ua);
  
  const chromeMatch = ua.match(/Chrome\/(\d+\.\d+\.\d+\.\d+)/);
  if (chromeMatch) {
    console.log("Detected version from UA (may be frozen):", chromeMatch[1]);
    // Check if it looks frozen (ends in .0.0.0)
    if (chromeMatch[1].match(/^\d+\.0\.0\.0$/)) {
      console.log("Warning: Version appears frozen, Client Hints may not be available");
    }
    return chromeMatch[1];
  }
  
  console.log("Could not detect version");
  return null;
}

// Initialize the popup
async function initialize() {
  console.log('Popup initializing...');
  
  // Create the UI elements
  createUI();
  
  // Auto-detect version immediately on popup open
  const detectedVersion = await detectBrowserVersion();
  if (detectedVersion) {
    // Store the detected version
    chrome.storage.local.set({
      currentVersion: detectedVersion,
      versionAutoDetected: true,
      lastAutoDetect: new Date().toISOString()
    }, function() {
      console.log("Stored detected version:", detectedVersion);
      loadVersions();
    });
  } else {
    // Load whatever we have stored
    loadVersions();
  }
  
  // Add event listeners
  document.getElementById('save-version-btn').addEventListener('click', saveCurrentVersion);
  document.getElementById('check-now-btn').addEventListener('click', checkForUpdates);
  document.getElementById('download-btn').addEventListener('click', openDownloadPage);
  document.getElementById('auto-detect-btn').addEventListener('click', triggerAutoDetect);
  
  // Show pin instructions on first run
  chrome.storage.local.get(['firstRun'], function(result) {
    if (result.firstRun !== false) {
      showPinInstructions();
      chrome.storage.local.set({firstRun: false});
    }
  });
  
  console.log('Popup initialized');
}

// Create the UI dynamically
function createUI() {
  const container = document.getElementById('container') || document.body;
  
  // Add styles
  const style = document.createElement('style');
  style.textContent = `
    body {
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', sans-serif;
      width: 340px;
      margin: 0;
      padding: 16px;
      background-color: #f7f9fc;
      color: #333;
    }
    .header {
      display: flex;
      align-items: center;
      margin-bottom: 20px;
    }
    .logo {
      width: 48px;
      height: 48px;
      margin-right: 12px;
    }
    .title {
      font-size: 18px;
      font-weight: 600;
      margin: 0;
    }
    .subtitle {
      font-size: 12px;
      color: #666;
      margin: 4px 0 0 0;
    }
    .version-card {
      background: white;
      border-radius: 8px;
      padding: 16px;
      box-shadow: 0 2px 4px rgba(0,0,0,0.05);
      margin-bottom: 16px;
    }
    .version-header {
      font-size: 14px;
      font-weight: 600;
      margin: 0 0 12px 0;
      color: #444;
    }
    .version-row {
      display: flex;
      justify-content: space-between;
      align-items: center;
      margin-bottom: 12px;
    }
    .version-label {
      font-size: 13px;
      color: #666;
    }
    .version-value {
      font-size: 14px;
      font-weight: 500;
      background: #f1f5f9;
      padding: 4px 8px;
      border-radius: 4px;
    }
    .update-available {
      background: #d1fae5;
      color: #065f46;
    }
    .auto-detected {
      background: #dbeafe;
      color: #1e40af;
    }
    .input-row {
      display: flex;
      gap: 8px;
      margin-bottom: 12px;
    }
    input {
      flex: 1;
      padding: 8px;
      border: 1px solid #d1d5db;
      border-radius: 4px;
      font-size: 14px;
    }
    button {
      padding: 8px 16px;
      border: none;
      border-radius: 4px;
      font-size: 14px;
      font-weight: 500;
      cursor: pointer;
      transition: background 0.2s;
    }
    .btn-primary {
      background: #3b82f6;
      color: white;
    }
    .btn-primary:hover {
      background: #2563eb;
    }
    .btn-secondary {
      background: #e5e7eb;
      color: #374151;
    }
    .btn-secondary:hover {
      background: #d1d5db;
    }
    .btn-success {
      background: #10b981;
      color: white;
    }
    .btn-success:hover {
      background: #059669;
    }
    .btn-info {
      background: #0ea5e9;
      color: white;
    }
    .btn-info:hover {
      background: #0284c7;
    }
    .message {
      font-size: 14px;
      padding: 12px;
      border-radius: 6px;
      margin-bottom: 16px;
    }
    .message-success {
      background: #d1fae5;
      color: #065f46;
    }
    .message-warning {
      background: #fef3c7;
      color: #92400e;
    }
    .message-error {
      background: #fee2e2;
      color: #b91c1c;
    }
    .message-info {
      background: #dbeafe;
      color: #1e40af;
    }
    .footer {
      font-size: 12px;
      color: #6b7280;
      text-align: center;
      margin-top: 20px;
    }
    .hidden {
      display: none;
    }
    .detection-badge {
      font-size: 11px;
      padding: 2px 6px;
      border-radius: 3px;
      margin-left: 8px;
    }
    .badge-auto {
      background: #dbeafe;
      color: #1e40af;
    }
    .badge-manual {
      background: #fef3c7;
      color: #92400e;
    }
    .manual-override {
      margin-top: 12px;
      padding-top: 12px;
      border-top: 1px solid #e5e7eb;
    }
    .manual-override-toggle {
      font-size: 12px;
      color: #6b7280;
      cursor: pointer;
      text-decoration: underline;
    }
    .manual-override-toggle:hover {
      color: #374151;
    }
  `;
  document.head.appendChild(style);
  
  // Create header
  const header = document.createElement('div');
  header.className = 'header';
  header.innerHTML = `
    <img src="icons/icon48.png" alt="Thorium Update Checker" class="logo">
    <div>
      <h1 class="title">Thorium Update Checker</h1>
      <p class="subtitle">by LXB Studio</p>
    </div>
  `;
  container.appendChild(header);
  
  // Create current version card with auto-detect info
  const currentVersionCard = document.createElement('div');
  currentVersionCard.className = 'version-card';
  currentVersionCard.innerHTML = `
    <h2 class="version-header">Your Thorium Version</h2>
    <div class="version-row">
      <span class="version-label">Detected:</span>
      <span>
        <span class="version-value" id="detected-version-display">Detecting...</span>
        <span class="detection-badge badge-auto" id="detection-badge">Auto</span>
      </span>
    </div>
    <button id="auto-detect-btn" class="btn-info" style="width: 100%; margin-bottom: 8px;">🔄 Re-detect Version</button>
    <div class="manual-override">
      <span class="manual-override-toggle" id="toggle-manual">Manual override (if auto-detect fails)</span>
      <div id="manual-input-area" class="hidden" style="margin-top: 8px;">
        <div class="input-row">
          <input type="text" id="current-version-input" placeholder="e.g. 138.0.7204.300">
          <button id="save-version-btn" class="btn-primary">Save</button>
        </div>
      </div>
    </div>
  `;
  container.appendChild(currentVersionCard);
  
  // Create latest version card
  const latestVersionCard = document.createElement('div');
  latestVersionCard.className = 'version-card';
  latestVersionCard.innerHTML = `
    <h2 class="version-header">Update Status</h2>
    <div class="version-row">
      <span class="version-label">Your Version:</span>
      <span class="version-value" id="current-version-display">Loading...</span>
    </div>
    <div class="version-row">
      <span class="version-label">Latest Version:</span>
      <span class="version-value" id="latest-version-display">Loading...</span>
    </div>
    <div class="version-row">
      <span class="version-label">Status:</span>
      <span class="version-value" id="update-status">Checking...</span>
    </div>
    <div class="version-row">
      <span class="version-label">Last checked:</span>
      <span id="last-checked">Just now</span>
    </div>
  `;
  container.appendChild(latestVersionCard);
  
  // Create action buttons
  const actionsDiv = document.createElement('div');
  actionsDiv.style.display = 'flex';
  actionsDiv.style.gap = '8px';
  
  const checkButton = document.createElement('button');
  checkButton.id = 'check-now-btn';
  checkButton.className = 'btn-secondary';
  checkButton.style.flex = '1';
  checkButton.textContent = 'Check Now';
  actionsDiv.appendChild(checkButton);
  
  const downloadButton = document.createElement('button');
  downloadButton.id = 'download-btn';
  downloadButton.className = 'btn-success';
  downloadButton.style.flex = '1';
  downloadButton.textContent = 'Download Update';
  actionsDiv.appendChild(downloadButton);
  
  container.appendChild(actionsDiv);
  
  // Create status message container (initially hidden)
  const messageDiv = document.createElement('div');
  messageDiv.id = 'message';
  messageDiv.className = 'message hidden';
  container.appendChild(messageDiv);
  
  // Create footer
  const footer = document.createElement('div');
  footer.className = 'footer';
  footer.innerHTML = `
    <p>Version auto-detected from browser</p>
    <p style="margin-top: 8px;">
      <span style="display: flex; align-items: center; justify-content: center; gap: 5px;">
        Made by <a href="https://lxb-studio.com" target="_blank" style="color: #3b82f6; text-decoration: none; font-weight: 500;">LXB Studio</a>
      </span>
    </p>
  `;
  container.appendChild(footer);
  
  // Add toggle for manual override
  document.getElementById('toggle-manual').addEventListener('click', function() {
    const manualArea = document.getElementById('manual-input-area');
    manualArea.classList.toggle('hidden');
    this.textContent = manualArea.classList.contains('hidden') 
      ? 'Manual override (if auto-detect fails)' 
      : 'Hide manual override';
  });
}

// Load current and latest versions
function loadVersions() {
  chrome.storage.local.get(['currentVersion', 'latestVersion', 'latestVersionInfo', 'versionAutoDetected'], function(result) {
    // Set current version
    const currentVersion = result.currentVersion || 'Not detected';
    const isAutoDetected = result.versionAutoDetected !== false;
    
    document.getElementById('detected-version-display').textContent = currentVersion;
    document.getElementById('current-version-display').textContent = currentVersion;
    document.getElementById('current-version-input').value = currentVersion !== 'Not detected' ? currentVersion : '';
    
    // Update detection badge
    const badge = document.getElementById('detection-badge');
    if (isAutoDetected && currentVersion !== 'Not detected') {
      badge.textContent = 'Auto';
      badge.className = 'detection-badge badge-auto';
    } else {
      badge.textContent = 'Manual';
      badge.className = 'detection-badge badge-manual';
    }
    
    // Set latest version if available
    if (result.latestVersion) {
      const latestVersion = result.latestVersion;
      document.getElementById('latest-version-display').textContent = latestVersion;
      
      // Check if update is available
      const hasUpdate = currentVersion !== 'Not detected' && isNewerVersion(latestVersion, currentVersion);
      const updateStatus = document.getElementById('update-status');
      
      if (hasUpdate) {
        updateStatus.textContent = 'Update Available!';
        updateStatus.className = 'version-value update-available';
        document.getElementById('download-btn').classList.remove('hidden');
      } else {
        updateStatus.textContent = 'Up to Date';
        updateStatus.className = 'version-value';
        document.getElementById('download-btn').classList.add('hidden');
      }
      
      // Update last checked time
      if (result.latestVersionInfo && result.latestVersionInfo.lastChecked) {
        const lastChecked = new Date(result.latestVersionInfo.lastChecked);
        const timeAgo = getTimeAgo(lastChecked);
        document.getElementById('last-checked').textContent = timeAgo;
      }
    } else {
      // If no latest version, trigger a check
      checkForUpdates();
    }
  });
}

// Trigger auto-detection directly in popup
async function triggerAutoDetect() {
  document.getElementById('detected-version-display').textContent = 'Detecting...';
  
  const detectedVersion = await detectBrowserVersion();
  
  if (detectedVersion) {
    chrome.storage.local.set({
      currentVersion: detectedVersion,
      versionAutoDetected: true,
      lastAutoDetect: new Date().toISOString()
    }, function() {
      showMessage('Version auto-detected: ' + detectedVersion, 'success');
      loadVersions();
      // Also trigger update check
      chrome.runtime.sendMessage({ action: 'checkNow' });
    });
  } else {
    showMessage('Could not auto-detect version. Client Hints API may not be available.', 'warning');
    document.getElementById('manual-input-area').classList.remove('hidden');
    document.getElementById('detected-version-display').textContent = 'Detection failed';
  }
}

// Save the user's current version
function saveCurrentVersion() {
  const input = document.getElementById('current-version-input');
  const version = input.value.trim();
  
  // Validate version format
  if (!version.match(/^\d+(\.\d+){1,3}$/)) {
    showMessage('Please enter a valid version number (e.g. 138.0.7204.300)', 'error');
    return;
  }
  
  chrome.storage.local.set({
    currentVersion: version,
    versionAutoDetected: false
  }, function() {
    document.getElementById('current-version-display').textContent = version;
    document.getElementById('detected-version-display').textContent = version;
    showMessage('Version saved successfully!', 'success');
    
    // Update badge to show manual
    const badge = document.getElementById('detection-badge');
    badge.textContent = 'Manual';
    badge.className = 'detection-badge badge-manual';
    
    // Trigger update check
    chrome.runtime.sendMessage({ action: 'checkNow' });
    
    // Refresh to update status
    setTimeout(loadVersions, 1000);
  });
}

// Check for updates
function checkForUpdates() {
  // Show loading states
  document.getElementById('latest-version-display').textContent = 'Checking...';
  document.getElementById('update-status').textContent = 'Checking...';
  document.getElementById('update-status').className = 'version-value';
  
  // Send check request to background
  chrome.runtime.sendMessage({
    action: 'checkNow'
  });
  
  // Use a timeout to refresh the display after a short delay
  setTimeout(() => {
    loadVersions();
  }, 1500);
}

// Open the download page
function openDownloadPage() {
  chrome.tabs.create({ url: 'https://github.com/Alex313031/thorium/releases/latest' });
}

// Show pin instructions dialog
function showPinInstructions() {
  const dialog = document.createElement('div');
  dialog.style.cssText = `
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.8);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 1000;
  `;
  
  const content = document.createElement('div');
  content.style.cssText = `
    background: white;
    padding: 20px;
    border-radius: 8px;
    max-width: 80%;
    text-align: center;
  `;
  
  content.innerHTML = `
    <h3 style="margin-top: 0;">✨ Auto-Detection Enabled!</h3>
    <p>Your Thorium version is now auto-detected from the browser.</p>
    <p style="margin-top: 12px;"><strong>Pin This Extension:</strong></p>
    <ol style="text-align: left;">
      <li>Click the puzzle icon in your toolbar</li>
      <li>Find "Thorium Update Checker"</li>
      <li>Click the pin icon</li>
    </ol>
    <button id="pin-close" style="padding: 8px 16px; background: #3b82f6; color: white; border: none; border-radius: 4px; cursor: pointer;">Got it!</button>
  `;
  
  dialog.appendChild(content);
  document.body.appendChild(dialog);
  
  document.getElementById('pin-close').addEventListener('click', function() {
    dialog.remove();
  });
}

// Show a message to the user
function showMessage(text, type = 'success') {
  const messageElement = document.getElementById('message');
  messageElement.textContent = text;
  messageElement.className = `message message-${type}`;
  
  // Auto-hide after a few seconds
  setTimeout(() => {
    messageElement.classList.add('hidden');
  }, 5000);
}

// Check if latest version is newer than current
function isNewerVersion(latest, current) {
  const latestParts = latest.replace(/^[vM]/, '').split('.').map(Number);
  const currentParts = current.replace(/^[vM]/, '').split('.').map(Number);
  
  for (let i = 0; i < Math.max(latestParts.length, currentParts.length); i++) {
    const latestPart = latestParts[i] || 0;
    const currentPart = currentParts[i] || 0;
    
    if (latestPart > currentPart) return true;
    if (latestPart < currentPart) return false;
  }
  
  return false; // Versions are equal
}

// Format time ago string
function getTimeAgo(date) {
  const now = new Date();
  const diffMs = now - date;
  const diffSec = Math.floor(diffMs / 1000);
  const diffMin = Math.floor(diffSec / 60);
  const diffHour = Math.floor(diffMin / 60);
  const diffDay = Math.floor(diffHour / 24);
  
  if (diffDay > 0) {
    return diffDay === 1 ? 'Yesterday' : `${diffDay} days ago`;
  } else if (diffHour > 0) {
    return `${diffHour} ${diffHour === 1 ? 'hour' : 'hours'} ago`;
  } else if (diffMin > 0) {
    return `${diffMin} ${diffMin === 1 ? 'minute' : 'minutes'} ago`;
  } else {
    return 'Just now';
  }
}
