// Simple background script for Thorium Update Checker

// Auto-detect browser version from user agent
function detectBrowserVersion() {
  const ua = navigator.userAgent;
  // Look for Chrome/VERSION pattern (Thorium uses Chromium's version string)
  const chromeMatch = ua.match(/Chrome\/(\d+\.\d+\.\d+\.\d+)/);
  if (chromeMatch) {
    return chromeMatch[1];
  }
  return null;
}

// Auto-detect version and store it
function autoDetectAndStore() {
  const detectedVersion = detectBrowserVersion();
  if (detectedVersion) {
    chrome.storage.local.set({
      currentVersion: detectedVersion,
      versionAutoDetected: true,
      lastAutoDetect: new Date().toISOString()
    });
    console.log("Auto-detected Thorium version:", detectedVersion);
    return detectedVersion;
  }
  return null;
}

// Check for updates on startup
chrome.runtime.onStartup.addListener(function() {
  autoDetectAndStore();
  checkGitHubForUpdates();
});

// Also check on install
chrome.runtime.onInstalled.addListener(function() {
  autoDetectAndStore();
  checkGitHubForUpdates();
});

// Set up periodic update checking (once a day)
chrome.alarms.create('checkForUpdates', { periodInMinutes: 1440 });

// Handle alarms
chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === 'checkForUpdates') {
    autoDetectAndStore(); // Re-detect in case browser was updated
    checkGitHubForUpdates();
  }
});

// Check GitHub for latest Thorium version
function checkGitHubForUpdates() {
  console.log("Checking GitHub for updates...");
  
  // Simple fetch with basic error handling
  fetch('https://api.github.com/repos/Alex313031/thorium/releases/latest')
    .then(response => {
      if (!response.ok) {
        throw new Error(`GitHub API error: ${response.status}`);
      }
      return response.json();
    })
    .then(data => {
      if (!data.tag_name) {
        throw new Error('No version found in response');
      }
      
      // Store the latest version
      const latestVersion = data.tag_name.replace(/^[vM]/, '');
      
      chrome.storage.local.set({
        latestVersion: latestVersion,
        latestVersionInfo: {
          downloadUrl: data.html_url,
          releaseDate: data.published_at,
          changelog: data.body || 'No changelog available',
          lastChecked: new Date().toISOString()
        }
      });
      
      console.log("Latest version from GitHub:", latestVersion);
      
      // Check if we should show notification
      chrome.storage.local.get(['currentVersion', 'notifiedVersion'], function(result) {
        const currentVersion = result.currentVersion;
        
        if (currentVersion && isNewerVersion(latestVersion, currentVersion)) {
          // Only notify if we haven't already notified about this version
          if (result.notifiedVersion !== latestVersion) {
            showUpdateNotification(latestVersion);
            setBadgeForUpdate();
            
            // Remember we've notified about this version
            chrome.storage.local.set({ notifiedVersion: latestVersion });
          }
        } else {
          clearUpdateBadge();
        }
      });
    })
    .catch(error => {
      console.error("Error checking for updates:", error);
    });
}

// Simple version comparison
function isNewerVersion(latest, current) {
  const latestParts = latest.replace(/^[vM]/, '').split('.').map(Number);
  const currentParts = current.replace(/^[vM]/, '').split('.').map(Number);
  
  for (let i = 0; i < Math.max(latestParts.length, currentParts.length); i++) {
    const latestPart = latestParts[i] || 0;
    const currentPart = currentParts[i] || 0;
    
    if (latestPart > currentPart) return true;
    if (latestPart < currentPart) return false;
  }
  
  return false; // Versions are equal
}

// Show update notification
function showUpdateNotification(version) {
  chrome.notifications.create('thorium-update', {
    type: 'basic',
    iconUrl: 'icons/icon128.png',
    title: 'Thorium Update Available',
    message: `Version ${version} is now available! - LXB Studio`,
    buttons: [
      { title: 'Download Now' }
    ],
    priority: 2
  });
}

// Handle notification clicks
chrome.notifications.onButtonClicked.addListener((notificationId, buttonIndex) => {
  if (notificationId === 'thorium-update') {
    chrome.tabs.create({ url: 'https://github.com/Alex313031/thorium/releases/latest' });
  }
});

// Set badge for update
function setBadgeForUpdate() {
  chrome.action.setBadgeText({text: "!"});
  chrome.action.setBadgeBackgroundColor({color: "#FF0000"});
}

// Clear update badge
function clearUpdateBadge() {
  chrome.action.setBadgeText({text: ""});
}

// Listen for messages from popup
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === 'setCurrentVersion') {
    chrome.storage.local.set({
      currentVersion: message.version,
      versionAutoDetected: false // Mark as manually set
    }, function() {
      // Trigger update check after setting version
      checkGitHubForUpdates();
      sendResponse({success: true});
    });
    return true; // Keep channel open for async response
  }
  
  if (message.action === 'checkNow') {
    checkGitHubForUpdates();
    sendResponse({success: true});
  }
  
  if (message.action === 'autoDetect') {
    const version = autoDetectAndStore();
    sendResponse({success: !!version, version: version});
    return true;
  }
  
  if (message.action === 'getDetectedVersion') {
    const version = detectBrowserVersion();
    sendResponse({version: version});
    return true;
  }
});
